<?php
// URL JSON dat
$json_url = "http://10.0.0.XXX:8080/uzly.json";

// Výchozí bod pro filtrování nesmyslných pozic
$ref_lat = 50;
$ref_lon = 15;
$max_distance_km = 300;
$min_separation_km = 0.1; // 100 m

// Funkce na výpočet vzdálenosti dvou souřadnic (Haversinova formule)
function haversine($lat1, $lon1, $lat2, $lon2) {
    $earth_radius = 6371; // Poloměr Země v km
    $dLat = deg2rad($lat2 - $lat1);
    $dLon = deg2rad($lon2 - $lon1);
    $a = sin($dLat / 2) * sin($dLat / 2) +
         cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
         sin($dLon / 2) * sin($dLon / 2);
    $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
    return $earth_radius * $c;
}

// Funkce pro posun souřadnic o danou vzdálenost (v km) náhodným směrem
function offset_coordinates($lat, $lon, $distance_km) {
    $bearing = rand(0, 360); // Náhodný směr v stupních
    $earth_radius = 6371; // Poloměr Země v km

    $lat_rad = deg2rad($lat);
    $lon_rad = deg2rad($lon);
    $bearing_rad = deg2rad($bearing);

    $new_lat_rad = asin(sin($lat_rad) * cos($distance_km / $earth_radius) +
                         cos($lat_rad) * sin($distance_km / $earth_radius) * cos($bearing_rad));

    $new_lon_rad = $lon_rad + atan2(sin($bearing_rad) * sin($distance_km / $earth_radius) * cos($lat_rad),
                                    cos($distance_km / $earth_radius) - sin($lat_rad) * sin($new_lat_rad));

    return [rad2deg($new_lat_rad), rad2deg($new_lon_rad)];
}

$json_data = file_get_contents($json_url);
$data = json_decode($json_data, true);

$nodes = [];
$placed_positions = [];

if ($data) {
    foreach ($data as $id => $node) {
        if (isset($node['nodeinfo']['id'], $node['position']['latitude'], $node['position']['longitude'])) {
            $lat = $node['position']['latitude'] / 1e7;
            $lon = $node['position']['longitude'] / 1e7;
            $last_seen = strtotime($node['last_seen']);
            $is_old = (time() - $last_seen) > 86400; // Starší než 24 hodin?

            // Odfiltrovat body, které jsou dále než 300 km
            if (haversine($ref_lat, $ref_lon, $lat, $lon) > $max_distance_km) {
                continue;
            }

            // Pokud tato pozice už existuje, posuň ji o 100 m náhodným směrem, dokud nenajdeme volné místo
            while (true) {
                $collision = false;
                foreach ($placed_positions as $existing) {
                    if (haversine($lat, $lon, $existing['lat'], $existing['lon']) < $min_separation_km) {
                        // Kolize → posunout bod
                        [$lat, $lon] = offset_coordinates($lat, $lon, $min_separation_km);
                        $collision = true;
                        break;
                    }
                }
                if (!$collision) {
                    break;
                }
            }
            $placed_positions[] = ['lat' => $lat, 'lon' => $lon];

            // Sestavení popisku
            $info = "<b>ID:</b> {$id}<br>";
            if (!empty($node['nodeinfo']['longname'])) $info .= "<b>Longname:</b> {$node['nodeinfo']['longname']}<br>";
            if (!empty($node['nodeinfo']['shortname'])) $info .= "<b>Shortname:</b> {$node['nodeinfo']['shortname']}<br>";
            if (!empty($node['nodeinfo']['hops_away'])) $info .= "<b>Hops away:</b> {$node['nodeinfo']['hops_away']}<br>";
            if (!empty($node['telemetry']['battery_level'])) $info .= "<b>Battery:</b> {$node['telemetry']['battery_level']}%<br>";
            if (!empty($node['telemetry']['temperature'])) $info .= "<b>Temperature:</b> {$node['telemetry']['temperature']}°C<br>";
            if (!empty($node['telemetry']['voltage'])) $info .= "<b>Voltage:</b> {$node['telemetry']['voltage']}V<br>";

            // Formátování času
            $info .= "<b>Last seen:</b> " . date("H:i:s d.m.Y", $last_seen) . "<br>";

            // Přidání do pole uzlů
            $nodes[] = [
                'lat' => $lat,
                'lon' => $lon,
                'info' => $info,
                'color' => $is_old ? 'red' : 'blue'
            ];
        }
    }
}
?>

<!DOCTYPE html>
<html lang="cs">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Meshtastic Map</title>
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css">
    <style>
        html, body, #map { height: 100%; margin: 0; padding: 0; }
    </style>
</head>
<body>
    <div id="map"></div>

    <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
    <script>
        var map = L.map('map');

        L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
            attribution: '&copy; OpenStreetMap contributors'
        }).addTo(map);

        var nodes = <?php echo json_encode($nodes); ?>;
        var markers = L.featureGroup();

        nodes.forEach(node => {
            var icon = L.icon({
                iconUrl: node.color === 'red' ? 'https://maps.google.com/mapfiles/ms/icons/red-dot.png' : 'https://maps.google.com/mapfiles/ms/icons/blue-dot.png',
                iconSize: [32, 32],
                iconAnchor: [16, 32],
                popupAnchor: [0, -32]
            });

            var marker = L.marker([node.lat, node.lon], { icon: icon })
                .bindPopup(node.info)
                .addTo(markers);
        });

        markers.addTo(map);
        map.fitBounds(markers.getBounds(), { padding: [20, 20] });
    </script>
</body>
</html>
